// background.js

// Store detected videos per tab
let detectedVideos = {};

const VIDEO_EXTENSIONS = ['.m3u8', '.mp4', '.flv', '.mov', '.mkv', '.webm', '.avi', '.wmv'];
const VIDEO_MIME_TYPES = [
    'application/x-mpegurl',
    'application/vnd.apple.mpegurl',
    'video/mp4',
    'video/webm',
    'video/ogg',
    'application/dash+xml'
];

// Resource states for hybrid filtering
const URL_TYPE = {
    VALID_VIDEO: 'VALID_VIDEO',
    BLACKLISTED: 'BLACKLISTED',
    UNCERTAIN: 'UNCERTAIN'
};

// Global Rule IDs
const RULES = {
    TIKTOK_BG: 60001,
    DOUYIN_BG: 60002
};

const UA = "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/126.0.0.0 Safari/537.36";

/**
 * Initialize Global DNR Rules for Background Fetches
 * This ensures that PROXY_FETCH (which runs in background) always has correct headers.
 */
async function setupGlobalRules() {
    const rules = [
        {
            id: RULES.TIKTOK_BG,
            priority: 1,
            action: {
                type: 'modifyHeaders',
                requestHeaders: [
                    { header: 'Referer', operation: 'set', value: 'https://www.tiktok.com/' },
                    { header: 'Origin', operation: 'set', value: 'https://www.tiktok.com' },
                    { header: 'User-Agent', operation: 'set', value: UA },
                    { header: 'Sec-Fetch-Site', operation: 'set', value: 'cross-site' },
                    { header: 'Sec-Fetch-Mode', operation: 'set', value: 'cors' },
                    { header: 'Sec-Fetch-Dest', operation: 'set', value: 'video' }
                ],
                responseHeaders: [
                    { header: 'Access-Control-Allow-Origin', operation: 'set', value: '*' },
                    { header: 'Access-Control-Allow-Methods', operation: 'set', value: 'GET, POST, OPTIONS, HEAD, PUT' },
                    { header: 'Access-Control-Allow-Headers', operation: 'set', value: '*' }
                ]
            },
            condition: {
                // Match tiktok.com, tiktokcdn.com, tiktokv.com
                regexFilter: "^https?://.*(tiktok|tiktokcdn|tiktokv)\\.com/.*",
                resourceTypes: ['xmlhttprequest', 'media', 'other']
            }
        },
        {
            id: RULES.DOUYIN_BG,
            priority: 1,
            action: {
                type: 'modifyHeaders',
                requestHeaders: [
                    { header: 'Referer', operation: 'set', value: 'https://www.douyin.com/' },
                    { header: 'Origin', operation: 'set', value: 'https://www.douyin.com' },
                    { header: 'User-Agent', operation: 'set', value: UA },
                    { header: 'Sec-Fetch-Site', operation: 'set', value: 'cross-site' },
                    { header: 'Sec-Fetch-Mode', operation: 'set', value: 'cors' },
                    { header: 'Sec-Fetch-Dest', operation: 'set', value: 'video' }
                ],
                responseHeaders: [
                    { header: 'Access-Control-Allow-Origin', operation: 'set', value: '*' },
                    { header: 'Access-Control-Allow-Methods', operation: 'set', value: 'GET, POST, OPTIONS, HEAD, PUT' },
                    { header: 'Access-Control-Allow-Headers', operation: 'set', value: '*' }
                ]
            },
            condition: {
                // Match douyin.com, douyinvod.com, amemv.com
                regexFilter: "^https?://.*(douyin|douyinvod|amemv)\\.com/.*",
                resourceTypes: ['xmlhttprequest', 'media', 'other']
            }
        }
    ];

    await chrome.declarativeNetRequest.updateSessionRules({
        removeRuleIds: [RULES.TIKTOK_BG, RULES.DOUYIN_BG],
        addRules: rules
    });
    console.log("Global DNR Rules initialized for TikTok/Douyin");
}

setupGlobalRules();

/**
 * Categorize URL based on its extension
 */
function checkUrlType(url) {
    if (!url || typeof url !== 'string') return URL_TYPE.BLACKLISTED;

    try {
        const urlObj = new URL(url);
        const pathname = urlObj.pathname.toLowerCase().replace(/\/+$/, '').trim();

        const WHITELIST = ['.m3u8', '.mp4', '.flv', '.mov', '.mkv', '.webm', '.avi', '.wmv'];
        const BLACKLIST = [
            '.js', '.css', '.map', '.ts', '.m4s', '.m4a', '.m4v', '.aac',
            '.jpg', '.jpeg', '.png', '.gif', '.webp', '.svg', '.ico',
            '.woff', '.woff2', '.ttf', '.otf', '.eot',
            '.json', '.xml', '.txt', '.pdf', '.html', '.htm', '.jsp', '.php', '.asp',
            '.scss', '.less', '.manifest'
        ];

        if (WHITELIST.some(ext => pathname.endsWith(ext))) return URL_TYPE.VALID_VIDEO;
        if (BLACKLIST.some(ext => pathname.endsWith(ext))) return URL_TYPE.BLACKLISTED;

        return URL_TYPE.UNCERTAIN;
    } catch (e) {
        return URL_TYPE.UNCERTAIN;
    }
}

/**
 * Utility to check if URL is a video (Synchronous check used for gates)
 */
function isVideoUrl(url) {
    const type = checkUrlType(url);
    // VALID_VIDEO is always ok. UNCERTAIN might be ok (Douyin), but we filter that in the sniffer.
    // For synchronous gates (like DOM scanner), we trust UNCERTAIN strings if they come from video tags.
    return type !== URL_TYPE.BLACKLISTED;
}

/**
 * Get filtered videos for a tab based on current settings
 */
async function getFilteredVideos(tabId) {
    if (!detectedVideos[tabId]) return [];

    const { minSize = 0 } = await chrome.storage.local.get('minSize');
    const minSizeBytes = parseFloat(minSize) * 1024 * 1024;

    const videos = Array.from(detectedVideos[tabId].values());

    return videos.filter(video => {
        // 1. Strict Extension/URL check
        if (!isVideoUrl(video.url)) return false;

        // 2. Size Filter
        // HLS streams (.m3u8) bypass the size filter
        const isStream = video.url.toLowerCase().split('?')[0].endsWith('.m3u8');

        if (!isStream && video.size && video.size < minSizeBytes) {
            return false;
        }

        return true;
    });
}

/**
 * Update the extension badge for a specific tab
 */
async function updateBadge(tabId) {
    const filteredVideos = await getFilteredVideos(tabId);
    const count = filteredVideos.length;
    chrome.action.setBadgeText({ tabId, text: count > 0 ? String(count) : '' });
}

/**
 * Add video to storage
 */
function addVideo(tabId, url, type = 'network', metadata = {}) {
    if (!url || !isVideoUrl(url)) return;

    if (!detectedVideos[tabId]) {
        detectedVideos[tabId] = new Map(); // Use Map to avoid duplicates
    }

    // No specialized deduplication needed for now

    // Use URL as key
    if (!detectedVideos[tabId].has(url)) {
        detectedVideos[tabId].set(url, {
            url,
            type,
            timestamp: Date.now(),
            ...metadata
        });

        updateBadge(tabId);
    }
}

// 1. Listen for Network Requests (Sniffing)
chrome.webRequest.onResponseStarted.addListener(
    (details) => {
        const { tabId, url, responseHeaders, type: resourceType } = details;
        if (tabId < 0) return;

        // IGNORE obviously non-video resource types
        // Enhanced list to catch more leaks
        const BLOCKED_TYPES = [
            'image', 'stylesheet', 'script', 'font', 'ping', 'csp_report',
            'beacon', 'sub_frame', 'main_frame'
        ];
        if (BLOCKED_TYPES.includes(resourceType)) {
            return;
        }

        const metadata = {};

        // Check categorization
        const urlCategory = checkUrlType(url);
        if (urlCategory === URL_TYPE.BLACKLISTED) return;

        // Check content-type header
        let isVideo = false;
        const contentTypeHeader = responseHeaders.find(h => h.name.toLowerCase() === 'content-type');

        if (contentTypeHeader) {
            const contentType = contentTypeHeader.value.toLowerCase();

            // 1. MUST NOT BE IMAGE
            if (contentType.startsWith('image/')) return;

            // 2. CHECK VIDEO MIME TYPES
            if (VIDEO_MIME_TYPES.some(type => contentType.includes(type)) || contentType.startsWith('video/')) {
                isVideo = true;
            }
        }

        // Final Decision logic:
        // - If it has a WHITELIST extension -> ACCEPT
        // - If it's UNCERTAIN but has a VIDEO MIME type -> ACCEPT (Douyin case)
        // - If it's UNCERTAIN and has NO MIME match -> Check extension fallback
        const finalDecision = (urlCategory === URL_TYPE.VALID_VIDEO) || (isVideo);

        if (finalDecision) {
            let processedUrl = url;

            // Extract content-length if available
            const contentLengthHeader = responseHeaders.find(h => h.name.toLowerCase() === 'content-length');
            let size = 0;
            if (contentLengthHeader) {
                size = parseInt(contentLengthHeader.value, 10) || 0;
            }

            addVideo(tabId, processedUrl, 'network', { from: 'sniffer', ...metadata, size });
        }
    },
    { urls: ["<all_urls>"] },
    ["responseHeaders"]
);

// 2. Listen for Messages
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
    if (message.action === 'GET_VIDEOS') {
        const tabId = message.tabId;
        getFilteredVideos(tabId).then(videos => {
            sendResponse({ videos });
        });
        return true; // Keep channel open for async response
    }

    if (message.action === 'HEARTBEAT') {
        sendResponse({ status: "ok" });
        return false; // Synchronous
    }

    if (message.action === 'FOUND_VIDEO_IN_DOM') {
        const { url, title } = message.payload;
        if (sender.tab) {
            addVideo(sender.tab.id, url, 'dom', { title, from: 'dom' });
        }
        // No response needed
        return false;
    }

    if (message.action === 'DOWNLOAD_AND_BYPASS') {
        handleDownloadAndBypass(message.payload);
        // No response needed
        return false;
    }

    if (message.action === 'PROXY_FETCH') {
        handleProxyFetch(message.payload).then(response => {
            sendResponse(response);
        });
        return true; // Keep channel open
    }

    return false; // Default: closed
});

/**
 * Handle Proxy Fetch from website
 */
async function handleProxyFetch({ url, options = {} }) {
    try {
        const fetchOptions = {
            method: options.method || 'GET',
            headers: options.headers || {},
            credentials: 'include' // Support potential cookie/session tied tokens
        };

        // Note: DNR rules set above will handle Referer/Origin/UA injection for TikTok/Douyin
        // We only handle generic Referer here if passed explicitly
        const refererKey = Object.keys(fetchOptions.headers).find(k => k.toLowerCase() === 'referer');
        if (refererKey) {
            fetchOptions.referrer = fetchOptions.headers[refererKey];
            fetchOptions.referrerPolicy = 'unsafe-url';
            delete fetchOptions.headers[refererKey];
        }

        const response = await fetch(url, fetchOptions);

        // Convert headers to plain object
        const responseHeaders = {};
        response.headers.forEach((value, key) => {
            responseHeaders[key] = value;
        });

        // Get body as blob/arrayBuffer then base64
        const blob = await response.blob();

        return new Promise((resolve) => {
            const reader = new FileReader();
            reader.onloadend = () => {
                const base64Data = reader.result.split(',')[1];
                resolve({
                    success: true,
                    status: response.status,
                    statusText: response.statusText,
                    headers: responseHeaders,
                    data: base64Data
                });
            };
            reader.onerror = () => {
                resolve({ success: false, error: 'Failed to read blob' });
            };
            reader.readAsDataURL(blob);
        });

    } catch (error) {
        console.error("Proxy Fetch Error:", error);
        return { success: false, error: error.message };
    }
}

// 3. Clear storage when tab closes or navigates
chrome.tabs.onRemoved.addListener((tabId) => {
    if (detectedVideos[tabId]) {
        delete detectedVideos[tabId];
    }
});

// Clear on page navigation/refresh
chrome.tabs.onUpdated.addListener((tabId, changeInfo) => {
    // Treat 'loading' and URL changes as reset points
    if (changeInfo.status === 'loading' || changeInfo.url) {
        delete detectedVideos[tabId];
        updateBadge(tabId);
    }
});

// React to global filter changes
chrome.storage.onChanged.addListener((changes) => {
    if (changes.minSize) {
        // Refresh all tab badges when minSize filter changes
        Object.keys(detectedVideos).forEach(tabId => {
            updateBadge(parseInt(tabId, 10));
        });
    }
});

/**
 * Handle Download:
 * 1. Extract Origin/Referer from the video URL (or use the page URL if passed).
 * 2. Set Dynamic Rule for DNR.
 * 3. Open the download page.
 */
// Track active download tabs to clean up rules
const downloadTabs = new Set();

/**
 * Handle Download:
 * 1. Create the target tab immediately.
 * 2. Set Session Rules for this specific tab ID to bypass CORS and spoof headers.
 * 3. The rules match ALL requests from this tab (*), ensuring redirects are covered.
 */
async function handleDownloadAndBypass({ videoUrl, referer, targetUrl }) {
    console.log('Preparing download for:', videoUrl);

    try {
        // 1. Create the tab first to get the ID
        const tab = await chrome.tabs.create({ url: targetUrl, active: true });
        const tabId = tab.id;
        downloadTabs.add(tabId);

        const urlObj = new URL(videoUrl);
        const refererValue = referer || `${urlObj.protocol}//${urlObj.host}/`;
        const originValue = new URL(refererValue).origin;

        // Rule IDs
        // Fixed: Use simple integer arithmetic and parseInt to avoid "expected integer, found number" errors.
        // We add a safety offset to avoid colliding with any old static/dynamic rules (e.g. 1001).
        const safeTabId = parseInt(tabId, 10);
        const ruleId = safeTabId + 50000;

        const requestHeaders = [
            { header: 'Referer', operation: 'set', value: refererValue },
            { header: 'Origin', operation: 'set', value: originValue },
            { header: 'User-Agent', operation: 'set', value: UA },
            { header: 'Sec-Fetch-Site', operation: 'set', value: 'cross-site' },
            { header: 'Sec-Fetch-Mode', operation: 'set', value: 'cors' },
            { header: 'Sec-Fetch-Dest', operation: 'set', value: 'video' }
        ];

        const addRules = [
            {
                id: ruleId,
                priority: 9999,
                action: {
                    type: 'modifyHeaders',
                    requestHeaders: requestHeaders,
                    responseHeaders: [
                        { header: 'Access-Control-Allow-Origin', operation: 'set', value: '*' },
                        { header: 'Access-Control-Allow-Methods', operation: 'set', value: 'GET, POST, OPTIONS, HEAD, PUT' },
                        { header: 'Access-Control-Allow-Headers', operation: 'set', value: '*' },
                        { header: 'Access-Control-Expose-Headers', operation: 'set', value: '*' },
                        { header: 'Cross-Origin-Resource-Policy', operation: 'set', value: 'cross-origin' }
                    ]
                },
                condition: {
                    tabIds: [safeTabId],
                    resourceTypes: ['xmlhttprequest', 'media', 'other', 'image', 'sub_frame']
                }
            }
        ];

        console.log(`Applying Session Rules for Tab ${safeTabId} (Rule ID: ${ruleId})`);

        // Use session rules - they are faster and don't persist to disk
        await chrome.declarativeNetRequest.updateSessionRules({
            addRules: addRules
        });

    } catch (e) {
        console.error("DNR Error:", e);
    }
}

// Clean up rules when a download tab is closed
chrome.tabs.onRemoved.addListener((tabId) => {
    if (downloadTabs.has(tabId)) {
        console.log(`Download tab ${tabId} closed. Cleaning up rules.`);
        downloadTabs.delete(tabId);

        try {
            const safeTabId = parseInt(tabId, 10);
            const ruleId = safeTabId + 50000;

            chrome.declarativeNetRequest.updateSessionRules({
                removeRuleIds: [ruleId]
            });
        } catch (e) {
            console.error("Error cleaning up rules:", e);
        }
    }
});
