// content.js

function scanForVideos() {
    const videos = document.querySelectorAll('video');
    videos.forEach(video => {
        let src = video.src;
        if (!src && video.querySelector('source')) {
            src = video.querySelector('source').src;
        }

        if (src && src.startsWith('http')) {
            try {
                // Strict extension check for DOM discoveries
                const urlObj = new URL(src);
                const pathname = urlObj.pathname.toLowerCase();

                const BLACKLIST = [
                    '.js', '.css', '.map', '.ts', '.jpg', '.jpeg', '.png', '.gif', '.webp', '.svg'
                ];

                // Block only explicitly blacklisted extensions
                const isBlacklisted = BLACKLIST.some(ext => pathname.endsWith(ext));

                if (!isBlacklisted) {
                    chrome.runtime.sendMessage({
                        action: 'FOUND_VIDEO_IN_DOM',
                        payload: {
                            url: src,
                            title: document.title
                        }
                    });
                }
            } catch (e) { }
        }
    });
}

// Initial scan
scanForVideos();

// Observe for new nodes
const observer = new MutationObserver((mutations) => {
    let shouldScan = false;
    mutations.forEach(mutation => {
        if (mutation.addedNodes.length) {
            shouldScan = true;
        }
    });
    if (shouldScan) {
        scanForVideos();
    }
});

observer.observe(document.body, { childList: true, subtree: true });

/**
 * Bridge between Webpage and Background script
 * Allows webpage to use extension features (like Proxy Fetch) without needing hardcoded extension ID
 */
window.addEventListener('message', (event) => {
    // Only accept messages from the same window
    if (event.source !== window) return;

    if (event.data && event.data.source === 'LIGHT_DOWNLOAD_WEB') {
        const { action, requestId, payload } = event.data;

        if (action === 'PING') {
            window.postMessage({
                source: 'LIGHT_DOWNLOAD_CONTENT',
                action: 'PONG',
                requestId: requestId
            }, '*');
            return;
        }

        // Generic relay to background
        chrome.runtime.sendMessage({ action, payload }, (response) => {
            // Handle lastError (e.g. extension context invalidated)
            if (chrome.runtime.lastError) {
                window.postMessage({
                    source: 'LIGHT_DOWNLOAD_CONTENT',
                    action: 'RESPONSE',
                    requestId: requestId,
                    response: { success: false, error: 'Extension context error: ' + chrome.runtime.lastError.message }
                }, '*');
                return;
            }

            // Send back response
            window.postMessage({
                source: 'LIGHT_DOWNLOAD_CONTENT',
                action: 'RESPONSE',
                requestId: requestId,
                response: response
            }, '*');
        });
    }
});
